<?php
/* Copyright (C) 2015		Jean-François Ferry     <jfefe@aternatik.fr>
 * Copyright (C) 2016		Laurent Destailleur		<eldy@users.sourceforge.net>
 * Copyright (C) 2018-2025  Frédéric France         <frederic.france@free.fr>
 * Copyright (C) 2025		MDW						<mdeweerd@users.noreply.github.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

use Luracast\Restler\RestException;

require_once DOL_DOCUMENT_ROOT.'/contrat/class/contrat.class.php';
require_once DOL_DOCUMENT_ROOT.'/societe/class/societe.class.php';

/**
 * API class for contracts
 *
 * @access protected
 * @class  DolibarrApiAccess {@requires user,external}
 */
class Contracts extends DolibarrApi
{
	/**
	 * @var string[]       Mandatory fields, checked when create and update object
	 */
	public static $FIELDS = array(
		'socid',
		'date_contrat',
		'commercial_signature_id',
		'commercial_suivi_id'
	);

	/**
	 * @var Contrat {@type Contrat}
	 */
	public $contract;

	/**
	 * Constructor
	 */
	public function __construct()
	{
		global $db;
		$this->db = $db;
		$this->contract = new Contrat($this->db);
	}

	/**
	 * Get properties of a contract object
	 *
	 * Return an array with contract information
	 *
	 * @param   int         $id         ID of contract
	 * @param 	string 		$properties Restrict the data returned to these properties. Ignored if empty. Comma separated list of properties names
	 * @param 	bool 		$withLines 	true or false to display or hide lines
	 * @return  Object					Object with cleaned properties
	 *
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 * @throws RestException 503 Error
	 */
	public function get($id, $properties = '', $withLines = true)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'lire')) {
			throw new RestException(403);
		}
		if ($id == 0) {
			throw new RestException(400, 'No contract with id=0 can exist');
		}
		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contract not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$this->contract->fetchObjectLinked();

		if (!$withLines) {
			unset($this->contract->lines);
		}

		return $this->_filterObjectProperties($this->_cleanObjectDatas($this->contract), $properties);
	}

	/**
	 * List contracts
	 *
	 * Get a list of contracts
	 *
	 * @param string		   $sortfield			Sort field
	 * @param string		   $sortorder			Sort order
	 * @param int			   $limit				Limit for list
	 * @param int			   $page				Page number
	 * @param string		   $thirdparty_ids		Thirdparty ids to filter contracts of (example '1' or '1,2,3') {@pattern /^[0-9,]*$/i}
	 * @param string           $sqlfilters          Other criteria to filter answers separated by a comma. Syntax example "(t.ref:like:'SO-%') and (t.date_creation:<:'20160101')"
	 * @param string		   $properties			Restrict the data returned to these properties. Ignored if empty. Comma separated list of properties names
	 * @param bool             $pagination_data     If this parameter is set to true the response will include pagination data. Default value is false. Page starts from 0*
	 * @param bool 			   $withLines 			true or false to display or hide lines
	 * @return  array                               Array of order objects
	 * @phan-return Contrat[]|array{data:Contrat[],pagination:array{total:int,page:int,page_count:int,limit:int}}
	 * @phpstan-return Contrat[]|array{data:Contrat[],pagination:array{total:int,page:int,page_count:int,limit:int}}
	 *
	 * @throws RestException 400 Bad Request
	 * @throws RestException 403 Access denied
	 * @throws RestException 503 Error
	 */
	public function index($sortfield = "t.rowid", $sortorder = 'ASC', $limit = 100, $page = 0, $thirdparty_ids = '', $sqlfilters = '', $properties = '', $pagination_data = false, $withLines = true)
	{
		global $db, $conf;

		if (!DolibarrApiAccess::$user->hasRight('contrat', 'lire')) {
			throw new RestException(403);
		}

		$obj_ret = array();

		// case of external user, $thirdparty_ids param is ignored and replaced by user's socid
		$socids = DolibarrApiAccess::$user->socid ?: $thirdparty_ids;

		// If the internal user must only see his customers, force searching by him
		$search_sale = 0;
		if (!DolibarrApiAccess::$user->hasRight('societe', 'client', 'voir') && !$socids) {
			$search_sale = DolibarrApiAccess::$user->id;
		}

		$sql = "SELECT t.rowid";
		$sql .= " FROM ".MAIN_DB_PREFIX."contrat AS t";
		$sql .= " INNER JOIN ".MAIN_DB_PREFIX."societe AS s ON (s.rowid = t.fk_soc)";
		$sql .= " LEFT JOIN ".MAIN_DB_PREFIX."contrat_extrafields AS ef ON (ef.fk_object = t.rowid)"; // Modification VMR Global Solutions to include extrafields as search parameters in the API GET call, so we will be able to filter on extrafields
		$sql .= ' WHERE t.entity IN ('.getEntity('contrat').')';
		if ($socids) {
			$sql .= " AND t.fk_soc IN (".$this->db->sanitize($socids).")";
		}
		// Search on sale representative
		if ($search_sale && $search_sale != '-1') {
			if ($search_sale == -2) {
				$sql .= " AND NOT EXISTS (SELECT sc.fk_soc FROM ".MAIN_DB_PREFIX."societe_commerciaux as sc WHERE sc.fk_soc = t.fk_soc)";
			} elseif ($search_sale > 0) {
				$sql .= " AND EXISTS (SELECT sc.fk_soc FROM ".MAIN_DB_PREFIX."societe_commerciaux as sc WHERE sc.fk_soc = t.fk_soc AND sc.fk_user = ".((int) $search_sale).")";
			}
		}
		// Add sql filters
		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		//this query will return total orders with the filters given
		$sqlTotals = str_replace('SELECT t.rowid', 'SELECT count(t.rowid) as total', $sql);

		$sql .= $this->db->order($sortfield, $sortorder);
		if ($limit) {
			if ($page < 0) {
				$page = 0;
			}
			$offset = $limit * $page;

			$sql .= $this->db->plimit($limit + 1, $offset);
		}

		$result = $this->db->query($sql);

		if ($result) {
			$num = $this->db->num_rows($result);
			$min = min($num, ($limit <= 0 ? $num : $limit));
			$i = 0;
			while ($i < $min) {
				$obj = $this->db->fetch_object($result);
				$contrat_static = new Contrat($this->db);
				if ($contrat_static->fetch($obj->rowid)) {
					if (!$withLines) {
						unset($contrat_static->lines);
					}
					$obj_ret[] = $this->_filterObjectProperties($this->_cleanObjectDatas($contrat_static), $properties);
				}
				$i++;
			}
		} else {
			throw new RestException(503, 'Error when retrieve contrat list : '.$this->db->lasterror());
		}

		//if $pagination_data is true the response will contain element data with all values and element pagination with pagination data(total,page,limit)
		if ($pagination_data) {
			$totalsResult = $this->db->query($sqlTotals);
			$total = $this->db->fetch_object($totalsResult)->total;

			$tmp = $obj_ret;
			$obj_ret = [];

			$obj_ret['data'] = $tmp;
			$obj_ret['pagination'] = [
				'total' => (int) $total,
				'page' => $page, //count starts from 0
				'page_count' => ceil((int) $total / $limit),
				'limit' => $limit
			];
		}

		return $obj_ret;
	}

	/**
	 * Create contract object
	 *
	 * @param   array   $request_data   Request data
	 * @phan-param ?array<string,string> $request_data
	 * @phpstan-param ?array<string,string> $request_data
	 * @return  int     ID of contrat
	 *
	 * @throws RestException 400 Bad Request
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 * @throws RestException 500 Error
	 */
	public function post($request_data = null)
	{
		global $conf;
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403, "Missing permission: Create/modify contracts/subscriptions");
		}

		$socid = (int) $request_data['socid'];
		$thirdpartytmp = new Societe($this->db);
		$thirdparty_result = $thirdpartytmp->fetch($socid);
		if ($thirdparty_result < 1) {
			throw new RestException(404, 'Thirdparty with id='.$socid.' not found or not allowed');
		}
		if (!DolibarrApi::_checkAccessToResource('societe', $thirdpartytmp->id)) {
			throw new RestException(404, 'Thirdparty with id='.$thirdpartytmp->id.' not found or not allowed');
		}

		// Check mandatory fields
		$result = $this->_validate($request_data);

		foreach ($request_data as $field => $value) {
			if ($field === 'caller') {
				// Add a mention of caller so on trigger called after action, we can filter to avoid a loop if we try to sync back again with the caller
				$this->contract->context['caller'] = sanitizeVal($request_data['caller'], 'aZ09');
				continue;
			}
			if ($field == 'id') {
				throw new RestException(400, 'Creating with id field is forbidden');
			}
			if ($field == 'entity' && ((int) $value) != ((int) $conf->entity)) {
				throw new RestException(403, 'Creating with entity='.((int) $value).' MUST be the same entity='.((int) $conf->entity).' as your API user/key belongs to');
			}

			if ($field == 'socid') {
				$thirdparty = new Societe($this->db);
				$result = $thirdparty->fetch((int) $value);
				if ($result < 1) {
					throw new RestException(404, 'Thirdparty with id='.((int) $value).' not found');
				}
			}

			$this->contract->$field = $this->_checkValForAPI($field, $value, $this->contract);
		}
		/*if (isset($request_data["lines"])) {
		  $lines = array();
		  foreach ($request_data["lines"] as $line) {
			array_push($lines, (object) $line);
		  }
		  $this->contract->lines = $lines;
		}*/
		if ($this->contract->create(DolibarrApiAccess::$user) < 0) {
			throw new RestException(500, "Error creating contract", array_merge(array($this->contract->error), $this->contract->errors));
		}

		return $this->contract->id;
	}

	/**
	 * Get lines of a contract
	 *
	 * @param int   	$id             	Id of contract
	 * @param string 	$sortfield			Sort field
	 * @param string	$sortorder			Sort order
	 * @param int		$limit				Limit for list
	 * @param int		$page				Page number
	 * @param string	$sqlfilters			Other criteria to filter answers separated by a comma. Syntax example "(t.ref:like:'SO-%') and (t.date_creation:<:'20160101')"
	 * @param string 	$properties 		Restrict the data returned to these properties. Ignored if empty. Comma separated list of properties names
	 * @param bool 		$pagination_data 	If this parameter is set to true the response will include pagination data. Default value is false. Page starts from 0*
	 * @return array						Array of contrat det objects
	 * @phan-return ContratLigne[]
	 * @phpstan-return ContratLigne[]
	 *
	 * @url	GET {id}/lines
	 *
	 * @throws RestException 404 Not Found
	 * @throws RestException 503 Error
	 */
	public function getLines($id, $sortfield = "d.rowid", $sortorder = 'ASC', $limit = 100, $page = 0, $sqlfilters = '', $properties = '', $pagination_data = false)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'lire')) {
			throw new RestException(403);
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contract not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$obj_ret = [];

		$sql = "SELECT d.rowid";
		$sql .= " FROM ".$this->db->prefix()."contratdet AS d";
		$sql .= " LEFT JOIN ".$this->db->prefix()."contrat AS c ON (c.rowid = d.fk_contrat)";
		$sql .= " LEFT JOIN ".$this->db->prefix()."contratdet_extrafields AS ef ON (ef.fk_object = d.rowid)";
		$sql .= " WHERE d.fk_contrat = ".((int) $id);
		$sql .= ' AND c.entity IN ('.getEntity('contrat').')';

		if ($sqlfilters) {
			$errormessage = '';
			$sql .= forgeSQLFromUniversalSearchCriteria($sqlfilters, $errormessage);
			if ($errormessage) {
				throw new RestException(400, 'Error when validating parameter sqlfilters -> '.$errormessage);
			}
		}

		$sqlTotals = str_replace('SELECT d.rowid', 'SELECT count(d.rowid) as total', $sql);

		$sql .= $this->db->order($sortfield, $sortorder);
		if ($limit) {
			if ($page < 0) {
				$page = 0;
			}
			$offset = $limit * $page;

			$sql .= $this->db->plimit($limit + 1, $offset);
		}

		$result = $this->db->query($sql);
		if ($result) {
			$num = $this->db->num_rows($result);
			$min = min($num, ($limit <= 0 ? $num : $limit));
			$i = 0;
			while ($i < $min) {
				$obj = $this->db->fetch_object($result);
				$contratdet_static = new ContratLigne($this->db);
				if ($contratdet_static->fetch($obj->rowid)) {
					$obj_ret[] = $this->_filterObjectProperties($this->_cleanObjectDatas($contratdet_static), $properties);
				}
				$i++;
			}
		} else {
			throw new RestException(503, 'Error when retrieve contratdet list : '.$this->db->lasterror());
		}

		if ($pagination_data) {
			$totalsResult = $this->db->query($sqlTotals);
			$total = $this->db->fetch_object($totalsResult)->total;

			$tmp = $obj_ret;
			$obj_ret = [];
			$obj_ret['data'] = $tmp;
			$obj_ret['pagination'] = [
				'total' => (int) $total,
				'page' => $page,
				'page_count' => ceil((int) $total / $limit),
				'limit' => $limit
			];
		}

		return $obj_ret;
	}

	/**
	 * Add a line to given contract
	 *
	 * @param int   $id             Id of contrat to update
	 * @param array $request_data   Contractline data
	 * @phan-param ?array<string,string> $request_data
	 * @phpstan-param ?array<string,string> $request_data
	 *
	 * @url	POST {id}/lines
	 *
	 * @return int|bool
	 */
	public function postLine($id, $request_data = null)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403);
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contract not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$request_data = (object) $request_data;

		$request_data->desc = sanitizeVal($request_data->desc, 'restricthtml');
		$request_data->price_base_type = sanitizeVal($request_data->price_base_type);

		$updateRes = $this->contract->addline(
			$request_data->desc,
			$request_data->subprice,
			$request_data->qty,
			$request_data->tva_tx,
			$request_data->localtax1_tx,
			$request_data->localtax2_tx,
			$request_data->fk_product,
			$request_data->remise_percent,
			$request_data->date_start,
			$request_data->date_end,
			$request_data->price_base_type ? $request_data->price_base_type : 'HT',
			$request_data->subprice_excl_tax,
			$request_data->info_bits,
			$request_data->fk_fournprice,
			$request_data->pa_ht,
			$request_data->array_options,
			$request_data->fk_unit,
			$request_data->rang
		);

		if ($updateRes > 0) {
			return $updateRes;
		}
		return false;
	}

	/**
	 * Update a line to given contract
	 *
	 * @param int   $id             Id of contrat to update
	 * @param int   $lineid         Id of line to update
	 * @param array $request_data   Contractline data
	 * @phan-param ?array<string,string> $request_data
	 * @phpstan-param ?array<string,string> $request_data
	 *
	 * @url	PUT {id}/lines/{lineid}
	 *
	 * @return Object|bool
	 */
	public function putLine($id, $lineid, $request_data = null)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403);
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contrat not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$request_data = (object) $request_data;

		$request_data->desc = sanitizeVal($request_data->desc, 'restricthtml');
		$request_data->price_base_type = sanitizeVal($request_data->price_base_type);

		$updateRes = $this->contract->updateline(
			$lineid,
			$request_data->desc,
			$request_data->subprice,
			$request_data->qty,
			$request_data->remise_percent,
			$request_data->date_start,
			$request_data->date_end,
			$request_data->tva_tx,
			$request_data->localtax1_tx,
			$request_data->localtax2_tx,
			$request_data->date_start_real,
			$request_data->date_end_real,
			$request_data->price_base_type ? $request_data->price_base_type : 'HT',
			$request_data->info_bits,
			$request_data->fk_fourn_price,
			$request_data->pa_ht,
			$request_data->array_options,
			$request_data->fk_unit
		);

		if ($updateRes > 0) {
			if (getDolGlobalInt('API_CONTRAT_PUTLINE_OUTPUT_LINE_ONLY')) {
				$result = new ContratLigne($this->db);
				$result->fetch($lineid);
				foreach (array(
					'array_languages',
					'contacts_ids',
					'linked_objects',
					'linkedObjectsIds',
					'actiontypecode',
					'module',
					'canvas',
					'user',
					'origin',
					'origin_id',
					'ref_ext',
					'status',
					'country_id',
					'country_code',
					'state_id',
					'region_id',
					'barcode_type',
					'barcode_type_coder',
					'mode_reglement_id',
					'cond_reglement_id',
					'demand_reason_id',
					'transport_mode_id',
					'shipping_method',
					'shipping_method_id',
					'model_pdf',
					'last_main_doc',
					'fk_bank',
					'fk_account',
					'lines',
					'name',
					'firstname',
					'lastname',
					'date_creation',
					'date_validation',
					'date_modification',
					'date_cloture',
					'user_author',
					'user_creation',
					'user_creation_id',
					'user_valid',
					'user_validation',
					'user_validation_id',
					'user_modification',
					'user_modification_id',
					'cond_reglement_supplier_id',
					'deposit_percent',
					'retained_warranty_fk_cond_reglement',
					'date_commande',
					'fk_user_creat',
					'fk_user_modif',
					'specimen',
					'fk_unit',
					'date_debut_prevue',
					'date_debut_reel',
					'date_fin_prevue',
					'date_fin_reel',
					'weight',
					'weight_units',
					'width',
					'width_units',
					'length',
					'length_units',
					'height',
					'height_units',
					'surface',
					'surface_units',
					'volume',
					'volume_units',
					'multilangs',
					'desc',
					'product',
					'fk_product_type',
					'warehouse_id',
					'totalpaid',
					'type',
					'libelle'
						 ) as $fieldToUnset) {
					unset($result->{$fieldToUnset});
				}
			} else {
				$result = $this->get($id);
				unset($result->line);
			}
			return $this->_cleanObjectDatas($result);
		} else {
			throw new RestException(500, implode(';', $this->contract->errors));
		}
	}

	/**
	 * Activate a service line of a given contract
	 *
	 * @param int		$id             Id of contract to activate
	 * @param int		$lineid         Id of line to activate
	 * @param string	$datestart		{@from body}  Date start        {@type timestamp}
	 * @param string    $dateend		{@from body}  Date end          {@type timestamp}
	 * @param string    $comment		{@from body}  Comment
	 *
	 * @url	PUT {id}/lines/{lineid}/activate
	 *
	 * @return Object|bool
	 *
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 */
	public function activateLine($id, $lineid, $datestart, $dateend = null, $comment = null)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'activer')) {
			throw new RestException(403);
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contrat not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$updateRes = $this->contract->active_line(DolibarrApiAccess::$user, $lineid, (int) $datestart, $dateend, $comment);

		if ($updateRes > 0) {
			$result = $this->get($id);
			unset($result->line);
			return $this->_cleanObjectDatas($result);
		}

		return false;
	}

	/**
	 * Unactivate a service line of a given contract
	 *
	 * @param int		$id             Id of contract to activate
	 * @param int		$lineid         Id of line to activate
	 * @param string	$datestart		{@from body}  Date start        {@type timestamp}
	 * @param string    $comment		{@from body}  Comment
	 *
	 * @url	PUT {id}/lines/{lineid}/unactivate
	 *
	 * @return Object|bool
	 *
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 */
	public function unactivateLine($id, $lineid, $datestart, $comment = null)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'desactiver')) {
			throw new RestException(403);
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contrat not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$updateRes = $this->contract->close_line(DolibarrApiAccess::$user, $lineid, (int) $datestart, $comment);

		if ($updateRes > 0) {
			$result = $this->get($id);
			unset($result->line);
			return $this->_cleanObjectDatas($result);
		}

		return false;
	}

	/**
	 * Delete a line to given contract
	 *
	 *
	 * @param int   $id             Id of contract to update
	 * @param int   $lineid         Id of line to delete
	 *
	 * @url	DELETE {id}/lines/{lineid}
	 *
	 * @return array|mixed
	 *
	 * @throws RestException 401
	 * @throws RestException 404
	 */
	public function deleteLine($id, $lineid)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403);
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contrat not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		// TODO Check the lineid $lineid is a line of object

		$updateRes = $this->contract->deleteLine($lineid, DolibarrApiAccess::$user);
		if ($updateRes > 0) {
			return $this->get($id);
		} else {
			throw new RestException(405, $this->contract->error);
		}
	}

	/**
	 * Update contract general fields (won't touch lines of contract)
	 *
	 * @param 	int   	$id             	Id of contract to update
	 * @param 	array 	$request_data   	Data
	 * @phan-param ?array<string,string> $request_data
	 * @phpstan-param ?array<string,string> $request_data
	 * @return 	Object						Updated object
	 *
	 * @throws RestException 400 Bad Request
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 * @throws RestException 500 Error
	 */
	public function put($id, $request_data = null)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403);
		}
		if ($id == 0) {
			throw new RestException(400, 'No contract with id=0 can exist');
		}

		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contrat not found');
		}

		$old_socid = $this->contract->socid;
		$oldthirdpartytmp = new Societe($this->db);
		$old_thirdparty_result = $oldthirdpartytmp->fetch($old_socid);
		if ($old_thirdparty_result < 1) {
			throw new RestException(404, 'Thirdparty with id='.$old_socid.' not found or not allowed');
		}
		if (!DolibarrApi::_checkAccessToResource('societe', $old_socid)) {
			throw new RestException(403, 'Access to old thirdparty='.$old_socid.' is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}
		foreach ($request_data as $field => $value) {
			if ($field == 'id') {
				throw new RestException(400, 'Updating with id field is forbidden');
			}
			if ($field == 'entity' && $value != $this->contract->entity) {
				throw new RestException(400, 'Changing entity of a contract using the APIs is not possible');
			}
			if ($field === 'caller') {
				// Add a mention of caller so on trigger called after action, we can filter to avoid a loop if we try to sync back again with the caller
				$this->contract->context['caller'] = sanitizeVal($request_data['caller'], 'aZ09');
				continue;
			}

			if ($field == 'socid') {
				$thirdparty = new Societe($this->db);
				$result = $thirdparty->fetch((int) $value);
				if ($result < 1) {
					throw new RestException(404, 'Thirdparty with id='.((int) $value).' not found');
				}
			}

			if ($field == 'array_options' && is_array($value)) {
				foreach ($value as $index => $val) {
					$this->contract->array_options[$index] = $this->_checkValForAPI($field, $val, $this->contract);
				}
				continue;
			}

			if ($field == 'socid') {
				$new_socid = (int) $value;
				$loopthirdpartytmp = new Societe($this->db);
				$new_thirdparty_result = $loopthirdpartytmp->fetch($new_socid);
				if ($new_thirdparty_result < 1) {
					throw new RestException(404, 'Thirdparty with id='.$new_socid.' not found or not allowed');
				}
				if (!DolibarrApi::_checkAccessToResource('societe', $new_socid)) {
					throw new RestException(403, 'Access to new thirdparty='.$new_socid.' is not allowed for login '.DolibarrApiAccess::$user->login);
				}
			}

			$this->contract->$field = $this->_checkValForAPI($field, $value, $this->contract);
		}

		if ($this->contract->update(DolibarrApiAccess::$user) > 0) {
			return $this->get($id);
		} else {
			throw new RestException(500, $this->contract->error);
		}
	}

	/**
	 * Delete contract
	 *
	 * @param   int     $id         Contract ID
	 *
	 * @return  array
	 * @phan-return array{success:array{code:int,message:string}}
	 * @phpstan-return array{success:array{code:int,message:string}}
	 *
	 * @throws RestException 400 Bad Request
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 * @throws RestException 500 Error
	 */
	public function delete($id)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'supprimer')) {
			throw new RestException(403, 'Missing permission: Delete contracts/subscriptions');
		}
		if ($id == 0) {
			throw new RestException(400, 'No contract with id=0 can exist');
		}
		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contract not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		if (!$this->contract->delete(DolibarrApiAccess::$user)) {
			throw new RestException(500, 'Error when delete contract : '.$this->contract->error);
		}

		return array(
			'success' => array(
				'code' => 200,
				'message' => 'Contract deleted'
			)
		);
	}

	/**
	 * Validate a contract
	 *
	 * @param   int $id             Contract ID
	 * @param   int $notrigger      1=Does not execute triggers, 0= execute triggers
	 *
	 * @url POST    {id}/validate
	 *
	 * @return  array
	 * @phan-return array{success:array{code:int,message:string}}
	 * @phpstan-return array{success:array{code:int,message:string}}
	 *
	 * FIXME An error 403 is returned if the request has an empty body.
	 * Error message: "Forbidden: Content type `text/plain` is not supported."
	 * Workaround: send this in the body
	 * {
	 *   "notrigger": 0
	 * }
	 *
	 * @throws RestException 304 Not Modified
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 * @throws RestException 500 Error
	 */
	public function validate($id, $notrigger = 0)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403);
		}
		if ($id == 0) {
			throw new RestException(400, 'No contract with id=0 can exist');
		}
		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contract not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$result = $this->contract->validate(DolibarrApiAccess::$user, '', $notrigger);
		if ($result == 0) {
			throw new RestException(304, 'Error nothing done. May be object is already validated');
		}
		if ($result < 0) {
			throw new RestException(500, 'Error when validating Contract: '.$this->contract->error);
		}

		return array(
			'success' => array(
				'code' => 200,
				'message' => 'Contract validated (Ref='.$this->contract->ref.')'
			)
		);
	}

	/**
	 * Close all services of a contract
	 *
	 * @param   int $id             Contract ID
	 * @param   int $notrigger      1=Does not execute triggers, 0= execute triggers
	 *
	 * @url POST    {id}/close
	 *
	 * @return  array
	 * @phan-return array{success:array{code:int,message:string}}
	 * @phpstan-return array{success:array{code:int,message:string}}
	 *
	 * FIXME An error 403 is returned if the request has an empty body.
	 * Error message: "Forbidden: Content type `text/plain` is not supported."
	 * Workaround: send this in the body
	 * {
	 *   "notrigger": 0
	 * }
	 *
	 * @throws RestException 304 Not Modified
	 * @throws RestException 403 Access denied
	 * @throws RestException 404 Not found
	 * @throws RestException 500 Error
	 */
	public function close($id, $notrigger = 0)
	{
		if (!DolibarrApiAccess::$user->hasRight('contrat', 'creer')) {
			throw new RestException(403);
		}
		if ($id == 0) {
			throw new RestException(400, 'No contract with id=0 can exist');
		}
		$result = $this->contract->fetch($id);
		if (!$result) {
			throw new RestException(404, 'Contract not found');
		}

		if (!DolibarrApi::_checkAccessToResource('contrat', $this->contract->id)) {
			throw new RestException(403, 'Access to this contract is not allowed for login '.DolibarrApiAccess::$user->login);
		}

		$result = $this->contract->closeAll(DolibarrApiAccess::$user, $notrigger);
		if ($result == 0) {
			throw new RestException(304, 'Error nothing done. May be object is already close');
		}
		if ($result < 0) {
			throw new RestException(500, 'Error when closing Contract: '.$this->contract->error);
		}

		return array(
			'success' => array(
				'code' => 200,
				'message' => 'Contract closed (Ref='.$this->contract->ref.'). All services were closed.'
			)
		);
	}



	// phpcs:disable PEAR.NamingConventions.ValidFunctionName.PublicUnderscore
	/**
	 * Clean sensible object datas
	 * @phpstan-template T
	 *
	 * @param   Object  $object     Object to clean
	 * @return  Object              Object with cleaned properties
	 * @phpstan-param T $object
	 * @phpstan-return T
	 */
	protected function _cleanObjectDatas($object)
	{
		// phpcs:enable
		$object = parent::_cleanObjectDatas($object);

		unset($object->address);
		unset($object->civility_id);

		return $object;
	}

	/**
	 * Validate fields before create or update object
	 *
	 * @param ?array<string,string> $data   Array with data to verify
	 * @return array<string,string>
	 *
	 * @throws RestException 400 Bad Request
	 */
	private function _validate($data)
	{
		if ($data === null) {
			$data = array();
		}
		$contrat = array();
		foreach (Contracts::$FIELDS as $field) {
			if (!isset($data[$field])) {
				throw new RestException(400, "$field field missing");
			}
			$contrat[$field] = $data[$field];
		}
		return $contrat;
	}
}
